%{

/*
 * parser.yy
 *
 * $ flex -t scanner.ll > scanner.cc
 * $ bison -d parser.yy -o parser.cc
 * $ mv parser.hh parser.h
 * $ g++ parser.cc scanner.cc -o parser
 */

#include <stdio.h>
#include "node.h"

/*
 * Reference yylex() and yytext from flex
 */
extern int yylex(void);
extern char *yytext;

/*
 * yyerror
 *
 * Used by bison to report error
 */
void yyerror(char *s) {
    fprintf(stderr, "%s: %s\n", s, yytext);
}

/*
 * display_postfix
 *
 * Display Postfix notation
 */
static void display_postfix(const Node *pcNode) {
    if (pcNode) {
        display_postfix(pcNode->left());
        display_postfix(pcNode->right());
        printf("%s ", pcNode->id().c_str());
    }
}

/*
 * free_nodes
 *
 * Free up memory allocated for node classes
 */
static void free_nodes(const Node *pcNode) {
    if (pcNode) {
        free_nodes(pcNode->left());
        free_nodes(pcNode->right());
        delete pcNode;
    }
}

%}

%union {
    string *id;
    Node   *node;
};

%token <id>    IDENT
%type  <node>  expr
%type  <node>  start

%%

start
: expr '\n' {
    // Display postfix notation from the parse tree
    display_postfix($1);
    printf("\n");
    // Free all the allocated memory
    free_nodes($$);
}
| start expr '\n' {
    // Display postfix notation from the parse tree
    display_postfix($2);
    printf("\n");
    // Free all the allocated memory
    free_nodes($$);
}
;

expr
: expr '+' expr {
    ($$ = new Node("+"))->add($1, $3);
}
| expr '*' expr {
    ($$ = new Node("*"))->add($1, $3);
}
| '(' expr ')' {
    $$ = $2;
}
| IDENT {
    $$ = new Node($1->c_str());
    delete $1;
}
;

%%

int main(void) {
    return (yyparse());
}
